<?php
/**
 * Privacy class; added to let customer export personal data
 *
 * @author  Your Inspiration Themes
 * @package YITH WooCommerce Affiliates
 * @version 1.2.5
 */

/*
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 */

if ( ! defined( 'YITH_WCAF' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCAF_Privacy' ) ) {
	/**
	 * YITH WCAF Privacy
	 *
	 * @since 1.2.5
	 */
	class YITH_WCAF_Privacy extends YITH_Privacy_Plugin_Abstract {

		/**
		 * Constructor method
		 *
		 * @return \YITH_WCAF_Privacy
		 * @since 1.2.5
		 */
		public function __construct() {
			parent::__construct( 'YITH WooCommerce Affiliates' );

			// set up affiliates data exporter
			add_filter( 'wp_privacy_personal_data_exporters', array( $this, 'register_exporter' ) );

			// set up affiliates data eraser
			add_filter( 'wp_privacy_personal_data_erasers', array( $this, 'register_eraser' ) );
		}

		/**
		 * Retrieves privacy example text for affiliates plugin
		 *
		 * @return string Privacy message
		 * @since 1.2.5
		 */
		public function get_privacy_message( $section ) {
			$content = '';

			switch ( $section ) {
				case 'collect_and_store':
					$content = '<p>' . __( 'While you visit our site, we’ll track:', 'yith-woocommerce-affiliates' ) . '</p>' .
							   '<ul>' .
							   '<li>' . __( 'Visits to the store: we’ll use this to generate statistics for affiliates and administrators.', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '<li>' . __( 'Location, IP address and browser type: we’ll use this just for statistics, and to let administrators supervise traffic generated by affiliates.', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '<li>' . __( 'Affiliate’s name, username, email address: we’ll use this information to register and keep track of affiliates.', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '</ul>' .
							   '<p>' . __( 'We’ll also use cookies to keep track of visit and sales generate by affiliates.', 'yith-woocommerce-affiliates' ) . '</p>';
					break;
				case 'has_access':
					$content = '<p>' . __( 'Members of our team have access to the information you provide us. For example, both Administrators and Shop Managers can access:', 'yith-woocommerce-affiliates' ) . '</p>' .
							   '<ul>' .
							   '<li>' . __( 'Affiliate’s personal details;', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '<li>' . __( 'System-generated commissions;', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '<li>' . __( 'Payments sent to the affiliates;', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '<li>' . __( 'Visits and sales generated through referral links.', 'yith-woocommerce-affiliates' ) . '</li>' .
							   '</ul>' .
							   '<p>' . __( 'Our team members have access to this information to correctly manage the affiliation system, and perform required actions in order to prevent customers from any inappropriate usage of the affiliate program.', 'yith-woocommerce-affiliates' ) . '</p>';
					break;
				case 'share':
					$content = '<p>' . __( 'We send payments through PayPal. When processing payments, some of your data will be passed to PayPal, including information required to process or support the payment, such as the commission total and payment information.', 'yith-woocommerce-affiliates' ) . '</p>' .
							   '<p>' . __( 'Please see the <a href="https://www.paypal.com/us/webapps/mpp/ua/privacy-full">PayPal Privacy Policy</a> for more details.', 'yith-woocommerce-affiliates' ) . '</p>';
					break;
				case 'payments':
				default:
					break;
			}

			return apply_filters( 'yith_wcaf_privacy_policy_content', $content, $section );
		}

		/**
		 * Register exporters for affiliates plugin
		 *
		 * @param $exporters array Array of currently registered exporters
		 *
		 * @return array Array of filtered exporters
		 * @since 1.2.5
		 */
		public function register_exporter( $exporters ) {

			// exports data about affiliate, and overall activity on the site
			$exporters['yith_wcaf_affiliate_details'] = array(
				'exporter_friendly_name' => __( 'Affiliate Data', 'yith-woocommerce-affiliates' ),
				'callback'               => array( $this, 'affiliate_details_exporter' )
			);

			// exports affiliate's commissions
			if ( apply_filters( 'yith_wcaf_privacy_export_commissions', false ) ) {
				$exporters['yith_wcaf_commissions'] = array(
					'exporter_friendly_name' => __( 'Affiliate Commissions', 'yith-woocommerce-affiliates' ),
					'callback'               => array( $this, 'commissions_exporter' )
				);
			}

			// exports affiliate's payments
			if ( apply_filters( 'yith_wcaf_privacy_export_payments', false ) ) {
				$exporters['yith_wcaf_payments'] = array(
					'exporter_friendly_name' => __( 'Affiliate Payments', 'yith-woocommerce-affiliates' ),
					'callback'               => array( $this, 'payments_exporter' )
				);
			}

			// exports affiliate's clicks
			if ( apply_filters( 'yith_wcaf_privacy_export_clicks', false ) ) {
				$exporters['yith_wcaf_clicks'] = array(
					'exporter_friendly_name' => __( 'Affiliate Clicks', 'yith-woocommerce-affiliates' ),
					'callback'               => array( $this, 'clicks_exporter' )
				);
			}

			return $exporters;
		}

		/**
		 * Register eraser for affiliate plugin
		 *
		 * @param $erasers array Array of currently registered erasers
		 *
		 * @return array Array of filtered erasers
		 * @since 1.2.5
		 */
		public function register_eraser( $erasers ) {
			// erases data about affiliate, and overall activity on the site
			$erasers['yith_wcaf_affiliate_details'] = array(
				'eraser_friendly_name' => __( 'Affiliate Data', 'yith-woocommerce-affiliates' ),
				'callback'             => array( $this, 'affiliate_details_eraser' )
			);

			// erases affiliate's commissions
			if ( apply_filters( 'yith_wcaf_privacy_erase_commissions', false ) ) {
				$erasers['yith_wcaf_commissions'] = array(
					'eraser_friendly_name' => __( 'Affiliate Commissions', 'yith-woocommerce-affiliates' ),
					'callback'             => array( $this, 'commissions_eraser' )
				);
			}

			// erases affiliate's payments
			if ( apply_filters( 'yith_wcaf_privacy_erase_payments', false ) ) {
				$erasers['yith_wcaf_payments'] = array(
					'eraser_friendly_name' => __( 'Affiliate Payments', 'yith-woocommerce-affiliates' ),
					'callback'             => array( $this, 'payments_eraser' )
				);
			}

			// erases affiliate's clicks
			if ( apply_filters( 'yith_wcaf_privacy_erase_clicks', false ) ) {
				$erasers['yith_wcaf_clicks'] = array(
					'eraser_friendly_name' => __( 'Affiliate Clicks', 'yith-woocommerce-affiliates' ),
					'callback'             => array( $this, 'clicks_eraser' )
				);
			}

			return $erasers;
		}

		/**
		 * Export affiliate details
		 *
		 * @param $email_address string Email of the users that requested export
		 *
		 * @return array Array of data to export
		 * @since 1.2.5
		 */
		public function affiliate_details_exporter( $email_address ) {
			$done           = true;
			$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$data_to_export = array();

			if ( $user instanceof WP_User ) {

				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

				if ( $affiliate ) {
					$personal_data = array();

					$props_to_export = apply_filters( 'yith_wcaf_privacy_export_affiliate_personal_data_props', array(
						'referral_link' => __( 'Referral link', 'yith-woocommerce-affiliates' ),
						'token'         => __( 'Token', 'yith-woocommerce-affiliates' ),
						'payment_email' => __( 'Payment Email', 'yith-woocommerce-affiliates' ),
						'earnings'      => __( 'Commissions earned', 'yith-woocommerce-affiliates' ),
						'paid'          => __( 'Total paid', 'yith-woocommerce-affiliates' )
					) );

					foreach ( $props_to_export as $prop => $description ) {
						$value = '';

						switch ( $prop ) {
							case 'referral_link':
								$referral_url = YITH_WCAF()->get_referral_url( $affiliate['token'] );

								$value = sprintf( '<a href="%1$s">%1$s</a>', $referral_url );
								break;
							case 'earnings':
								$value = wc_price( $affiliate['earnings'] );
								break;
							case 'paid':
								$value = wc_price( $affiliate['paid'] );
								break;
							default:
								if ( isset( $affiliate[ $prop ] ) ) {
									$value = $affiliate[ $prop ];
								}
								break;
						}

						$value = apply_filters( 'yith_wcaf_privacy_export_affiliate_personal_data_prop_value', $value, $prop, $affiliate );

						if ( $value ) {
							$personal_data[] = array(
								'name'  => $description,
								'value' => $value,
							);
						}
					}

					$data_to_export[] = array(
						'group_id'    => 'yith_wcwl_affiliate_data',
						'group_label' => __( 'Affiliate Data', 'yith-woocommerce-affiliates' ),
						'item_id'     => 'affiliate-' . $affiliate['ID'],
						'data'        => $personal_data,
					);
				}

			}

			return array(
				'data' => $data_to_export,
				'done' => $done,
			);
		}

		/**
		 * Export affiliate commissions
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Array of data to export
		 * @since 1.2.5
		 */
		public function commissions_exporter( $email_address, $page ) {
			$done           = true;
			$page           = (int) $page;
			$offset         = 50 * ( $page - 1 );
			$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$data_to_export = array();

			if ( $user instanceof WP_User ) {
				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

				if ( $affiliate ) {
					$commissions = YITH_WCAF_Commission_Handler()->get_commissions( array(
						'affiliate_id' => $affiliate['ID'],
						'limit'        => 50,
						'offset'       => $offset
					) );

					if ( 0 < count( $commissions ) ) {
						foreach ( $commissions as $commission ) {
							$data_to_export[] = array(
								'group_id'    => 'yith_wcaf_commissions',
								'group_label' => __( 'Affiliate Commissions', 'yith-woocommerce-affiliates' ),
								'item_id'     => 'commission-' . $commission['ID'],
								'data'        => $this->get_commission_data( $commission ),
							);
						}
						$done = 10 > count( $commissions );
					} else {
						$done = true;
					}
				}
			}

			return array(
				'data' => $data_to_export,
				'done' => $done,
			);
		}

		/**
		 * Export affiliate payments
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Array of data to export
		 * @since 1.2.5
		 */
		public function payments_exporter( $email_address, $page ) {
			$done           = true;
			$page           = (int) $page;
			$offset         = 50 * ( $page - 1 );
			$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$data_to_export = array();

			if ( $user instanceof WP_User ) {
				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

				if ( $affiliate ) {
					$payments = YITH_WCAF_Payment_Handler()->get_payments( array(
						'affiliate_id' => $affiliate['ID'],
						'limit'        => 50,
						'offset'       => $offset
					) );

					if ( 0 < count( $payments ) ) {
						foreach ( $payments as $payment ) {
							$data_to_export[] = array(
								'group_id'    => 'yith_wcaf_payments',
								'group_label' => __( 'Affiliate Payments', 'yith-woocommerce-affiliates' ),
								'item_id'     => 'payment-' . $payment['ID'],
								'data'        => $this->get_payment_data( $payment ),
							);
						}
						$done = 10 > count( $payments );
					} else {
						$done = true;
					}
				}
			}

			return array(
				'data' => $data_to_export,
				'done' => $done,
			);
		}

		/**
		 * Export affiliate clicks
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Array of data to export
		 * @since 1.2.5
		 */
		public function clicks_exporter( $email_address, $page ) {
			$done           = true;
			$page           = (int) $page;
			$offset         = 50 * ( $page - 1 );
			$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$data_to_export = array();

			if ( $user instanceof WP_User ) {
				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

				if ( $affiliate ) {
					$clicks = YITH_WCAF_Click_Handler()->get_hits( array(
						'affiliate_id' => $affiliate['ID'],
						'limit'        => 50,
						'offset'       => $offset
					) );

					if ( 0 < count( $clicks ) ) {
						foreach ( $clicks as $click ) {
							$data_to_export[] = array(
								'group_id'    => 'yith_wcaf_clicks',
								'group_label' => __( 'Affiliate Hits', 'yith-woocommerce-affiliates' ),
								'item_id'     => 'hit-' . $click['ID'],
								'data'        => $this->get_click_data( $click ),
							);
						}
						$done = 10 > count( $clicks );
					} else {
						$done = true;
					}
				}
			}

			return array(
				'data' => $data_to_export,
				'done' => $done,
			);
		}

		/**
		 * Erase affiliate personal details
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Result of the operation
		 */
		public function affiliate_details_eraser( $email_address ) {
			$user     = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$response = array(
				'items_removed'  => false,
				'items_retained' => false,
				'messages'       => array(),
				'done'           => true,
			);

			if ( ! $user instanceof WP_User ) {
				return $response;
			}

			$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

			if ( ! $affiliate ) {
				return $response;
			}

			$anonymized_data = array();
			$props_to_remove = apply_filters( 'yith_wcaf_privacy_remove_affiliate_personal_data_props', array(
				'payment_email' => 'email'
			) );

			if ( ! empty( $props_to_remove ) && is_array( $props_to_remove ) ) {
				foreach ( $props_to_remove as $prop => $data_type ) {
					$value = isset( $affiliate[ $prop ] ) ? $affiliate[ $prop ] : false;

					// If the value is empty, it does not need to be anonymized.
					if ( empty( $value ) || empty( $data_type ) ) {
						continue;
					}

					if ( function_exists( 'wp_privacy_anonymize_data' ) ) {
						$anon_value = wp_privacy_anonymize_data( $data_type, $value );
					} else {
						$anon_value = '';
					}

					$anonymized_data[ $prop ] = apply_filters( 'yith_wcaf_privacy_remove_affiliate_personal_data_prop_value', $anon_value, $prop, $value, $data_type, $affiliate );
				}

				YITH_WCAF_Affiliate_Handler()->update( $affiliate['ID'], $anonymized_data );

				$response['messages'][]    = sprintf( __( 'Anonymized affiliate %s.', 'yith-woocommerce-affiliates' ), $affiliate['ID'] );
				$response['items_removed'] = true;
			}

			return $response;
		}

		/**
		 * Erase affiliate's commissions
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Result of the operation
		 */
		public function commissions_eraser( $email_address, $page ) {
			$page     = (int) $page;
			$offset   = 10 * ( $page - 1 );
			$user     = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$response = array(
				'items_removed'  => false,
				'items_retained' => false,
				'messages'       => array(),
				'done'           => true,
			);

			if ( ! $user instanceof WP_User ) {
				return $response;
			}

			$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

			if ( ! $affiliate ) {
				return $response;
			}

			$commissions = YITH_WCAF_Commission_Handler()->get_commissions( array(
				'limit'        => 10,
				'offset'       => $offset,
				'affiliate_id' => $affiliate['ID'],
				'orderby'      => 'ID',
				'order'        => 'ASC'
			) );

			if ( 0 < count( $commissions ) ) {
				foreach ( $commissions as $commission ) {
					if ( apply_filters( 'yith_wcaf_privacy_erase_commission', true, $commission ) ) {
						do_action( 'yith_wcaf_privacy_before_remove_commission', $commission );

						if ( function_exists( 'wp_privacy_anonymize_data' ) ) {
							$anon_value = wp_privacy_anonymize_data( 'numeric_id', $commission['affiliate_id'] );
						} else {
							$anon_value = 0;
						}

						YITH_WCAF_Commission_Handler()->update( $commission['ID'], array(
							'affiliate_id' => $anon_value
						) );

						do_action( 'yith_wcaf_privacy_remove_commission', $commission );

						/* Translators: %s Order number. */
						$response['messages'][]    = sprintf( __( 'Anonymized commission %s.', 'yith-woocommerce-affiliates' ), $commission['ID'] );
						$response['items_removed'] = true;
					} else {
						/* Translators: %s Order number. */
						$response['messages'][]     = sprintf( __( 'Commission %s has been retained.', 'yith-woocommerce-affiliates' ), $commission['ID'] );
						$response['items_retained'] = true;
					}
				}
				$response['done'] = 10 > count( $commissions );
			} else {
				$response['done'] = true;
			}

			return $response;
		}

		/**
		 * Erase affiliate's payments
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Result of the operation
		 */
		public function payments_eraser( $email_address, $page ) {
			$page     = (int) $page;
			$offset   = 10 * ( $page - 1 );
			$user     = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$response = array(
				'items_removed'  => false,
				'items_retained' => false,
				'messages'       => array(),
				'done'           => true,
			);

			if ( ! $user instanceof WP_User ) {
				return $response;
			}

			$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

			if ( ! $affiliate ) {
				return $response;
			}

			$payments = YITH_WCAF_Payment_Handler()->get_payments( array(
				'limit'        => 10,
				'offset'       => $offset,
				'affiliate_id' => $affiliate['ID'],
				'orderby'      => 'ID',
				'order'        => 'ASC'
			) );

			if ( 0 < count( $payments ) ) {
				foreach ( $payments as $payment ) {
					if ( apply_filters( 'yith_wcaf_privacy_erase_payment', true, $payment ) ) {
						do_action( 'yith_wcaf_privacy_before_remove_payment', $payment );

						if ( function_exists( 'wp_privacy_anonymize_data' ) ) {
							$anon_value = wp_privacy_anonymize_data( 'numeric_id', $payment['affiliate_id'] );
						} else {
							$anon_value = 0;
						}

						YITH_WCAF_Payment_Handler()->update( $payment['ID'], array(
							'affiliate_id' => $anon_value
						) );

						do_action( 'yith_wcaf_privacy_remove_payment', $payment );

						/* Translators: %s Order number. */
						$response['messages'][]    = sprintf( __( 'Anonymized payment %s.', 'yith-woocommerce-affiliates' ), $payment['ID'] );
						$response['items_removed'] = true;
					} else {
						/* Translators: %s Order number. */
						$response['messages'][]     = sprintf( __( 'Payment %s has been retained.', 'yith-woocommerce-affiliates' ), $payment['ID'] );
						$response['items_retained'] = true;
					}
				}
				$response['done'] = 10 > count( $payments );
			} else {
				$response['done'] = true;
			}

			return $response;
		}

		/**
		 * Erase affiliate's clicks
		 *
		 * @param $email_address string Email of the users that requested export
		 * @param $page          int Current page processed
		 *
		 * @return array Result of the operation
		 */
		public function clicks_eraser( $email_address, $page ) {
			global $wpdb;

			$page     = (int) $page;
			$offset   = 10 * ( $page - 1 );
			$user     = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
			$response = array(
				'items_removed'  => false,
				'items_retained' => false,
				'messages'       => array(),
				'done'           => true,
			);

			if ( ! $user instanceof WP_User ) {
				return $response;
			}

			$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_user_id( $user->ID );

			if ( ! $affiliate ) {
				return $response;
			}

			$clicks = YITH_WCAF_Click_Handler()->get_hits( array(
				'limit'        => 10,
				'offset'       => $offset,
				'affiliate_id' => $affiliate['ID'],
				'orderby'      => 'ID',
				'order'        => 'ASC'
			) );

			if ( 0 < count( $clicks ) ) {
				foreach ( $clicks as $click ) {
					if ( apply_filters( 'yith_wcaf_privacy_erase_click', true, $click ) ) {
						do_action( 'yith_wcaf_privacy_before_remove_click', $click );

						if ( function_exists( 'wp_privacy_anonymize_data' ) ) {
							$anon_value = wp_privacy_anonymize_data( 'numeric_id', $click['affiliate_id'] );
						} else {
							$anon_value = 0;
						}

						$wpdb->update( $wpdb->yith_clicks, array( 'affiliate_id' => $anon_value ), array( 'ID' => $click['ID'] ) );

						do_action( 'yith_wcaf_privacy_remove_click', $click );

						/* Translators: %s Order number. */
						$response['messages'][]    = sprintf( __( 'Anonymized hit %s.', 'yith-woocommerce-affiliates' ), $click['ID'] );
						$response['items_removed'] = true;
					} else {
						/* Translators: %s Order number. */
						$response['messages'][]     = sprintf( __( 'Hit %s has been retained.', 'yith-woocommerce-affiliates' ), $click['ID'] );
						$response['items_retained'] = true;
					}
				}
				$response['done'] = 10 > count( $clicks );
			} else {
				$response['done'] = true;
			}

			return $response;
		}

		/**
		 * Retrieves data to export for each affiliate's commission
		 *
		 * @param $commission array Commission processed
		 *
		 * @return array Data to export
		 * @since 1.2.5
		 */
		protected function get_commission_data( $commission ) {
			$personal_data   = array();
			$props_to_export = apply_filters( 'yith_wcaf_privacy_export_commission_props', array(
				'ID'           => __( 'ID', 'yith-woocommerce-affiliates' ),
				'status'       => __( 'Status', 'yith-woocommerce-affiliates' ),
				'order_id'     => __( 'Order id', 'yith-woocommerce-affiliates' ),
				'product_name' => __( 'Product', 'yith-woocommerce-affiliates' ),
				'created_at'   => __( 'Created on', 'yith-woocommerce-affiliates' ),
				'amount'       => __( 'Amount', 'yith-woocommerce-affiliates' ),
			), $commission );

			foreach ( $props_to_export as $prop => $name ) {
				$value = '';

				switch ( $prop ) {
					case 'amount':
						$value = wc_price( $commission['amount'] );
						break;
					case 'status':
						$value = YITH_WCAF_Commission_Handler()->get_readable_status( $commission['status'] );
						break;
					case 'created_at':
						$date = $commission['created_at'];

						$value = date_i18n( 'd F Y', strtotime( $date ) );
						break;
					default:
						if ( isset( $commission[ $prop ] ) ) {
							$value = $commission[ $prop ];
						}
						break;
				}

				$value = apply_filters( 'yith_wcaf_privacy_export_commission_prop', $value, $prop, $commission );

				if ( $value ) {
					$personal_data[] = array(
						'name'  => $name,
						'value' => $value,
					);
				}
			}

			$personal_data = apply_filters( 'yith_wcaf_privacy_export_commission', $personal_data, $commission );

			return $personal_data;
		}

		/**
		 * Retrieves data to export for each affiliates's payment
		 *
		 * @param $payment array Payment processed
		 *
		 * @return array Data to export
		 * @since 1.2.5
		 */
		protected function get_payment_data( $payment ) {
			$personal_data   = array();
			$props_to_export = apply_filters( 'yith_wcaf_privacy_export_payment_props', array(
				'ID'            => __( 'ID', 'yith-woocommerce-affiliates' ),
				'status'        => __( 'Status', 'yith-woocommerce-affiliates' ),
				'payment_email' => __( 'Payment email', 'yith-woocommerce-affiliates' ),
				'created_at'    => __( 'Created on', 'yith-woocommerce-affiliates' ),
				'completed_at'  => __( 'Completed on', 'yith-woocommerce-affiliates' ),
				'amount'        => __( 'Amount', 'yith-woocommerce-affiliates' ),
				'gateway'       => __( 'Gateway', 'yith-woocommerce-affiliates' )

			), $payment );

			foreach ( $props_to_export as $prop => $name ) {
				$value = '';

				switch ( $prop ) {
					case 'status':
						$value = YITH_WCAF_Payment_Handler()->get_readable_status( $payment['status'] );
						break;
					case 'created_at':
						$date = $payment['created_at'];

						$value = date_i18n( 'd F Y', strtotime( $date ) );
						break;
					case 'completed_at':
						$date = $payment['completed_at'];

						$value = empty( $payment['completed_at'] ) ? __( 'N/A', 'yith-woocommerce-affiliates' ) : date_i18n( 'd F Y', strtotime( $date ) );
						break;
					case 'gateway':
						if ( defined( 'YITH_WCAF_Premium' ) && ! empty( $payment['gateway'] ) ) {
							$value = YITH_WCAF_Payment_Handler_Premium()->get_readable_gateway( $payment['gateway'] );
						}
						break;
					case 'amount':
						$value = wc_price( $payment['amount'] );
						break;
					default:
						if ( isset( $payment[ $prop ] ) ) {
							$value = $payment[ $prop ];
						}
						break;
				}

				$value = apply_filters( 'yith_wcaf_privacy_export_payment_prop', $value, $prop, $payment );

				if ( $value ) {
					$personal_data[] = array(
						'name'  => $name,
						'value' => $value,
					);
				}
			}

			$personal_data = apply_filters( 'yith_wcaf_privacy_export_payment', $personal_data, $payment );

			return $personal_data;
		}

		/**
		 * Retrieves data to export for each affiliates's hit
		 *
		 * @param $click array Click processed
		 *
		 * @return array Data to export
		 * @since 1.2.5
		 */
		protected function get_click_data( $click ) {
			$personal_data   = array();
			$props_to_export = apply_filters( 'yith_wcaf_privacy_export_click_props', array(
				'ID'         => __( 'ID', 'yith-woocommerce-affiliates' ),
				'link'       => __( 'Link', 'yith-woocommerce-affiliates' ),
				'origin'     => __( 'Origin', 'yith-woocommerce-affiliates' ),
				'click_date' => __( 'Date', 'yith-woocommerce-affiliates' ),
				'conv_date'  => __( 'Converted on', 'yith-woocommerce-affiliates' )

			), $click );

			foreach ( $props_to_export as $prop => $name ) {
				$value = '';

				switch ( $prop ) {
					case 'click_date':
						$date = $click['click_date'];

						$value = date_i18n( 'd F Y', strtotime( $date ) );
						break;
					case 'conv_date':
						$date = $click['conv_date'];

						$value = empty( $date ) ? __( 'N/A', 'yith-woocommerce-affiliates' ) : date_i18n( 'd F Y', strtotime( $date ) );
						break;
					default:
						if ( isset( $click[ $prop ] ) ) {
							$value = $click[ $prop ];
						}
						break;
				}

				$value = apply_filters( 'yith_wcaf_privacy_export_click_prop', $value, $prop, $click );

				if ( $value ) {
					$personal_data[] = array(
						'name'  => $name,
						'value' => $value,
					);
				}
			}

			$personal_data = apply_filters( 'yith_wcaf_privacy_export_click', $personal_data, $click );

			return $personal_data;
		}
	}
}